<?php
/**
 * Plugin Name: Hibiscus Image Optimizer
 * Description: A modern, premium plugin for bulk and on-upload image conversion to WebP/AVIF.
 * Version: 1.2.0
 * Author: Hibiscus Technolab
 * Author URI: https://hibiscustechnolab.com
 * Text Domain: hibiscus-image-optimizer
 * Plugin URI: https://hibiscustechnolab.com/hibiscus-image-optimization-plugin-for-wordpress/
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('HIO_VERSION', '1.0.0.' . time());
define( 'HIO_PATH', plugin_dir_path( __FILE__ ) );
define( 'HIO_URL', plugin_dir_url( __FILE__ ) );
define( 'HIO_SLUG', 'hibiscus-image-optimizer' ); // The plugin folder and menu slug
if ( ! function_exists( 'hio_fs' ) ) {
    // Create a helper function for easy SDK access.
    function hio_fs() {
        global $hio_fs;

        if ( ! isset( $hio_fs ) ) {
            // Include Freemius SDK.
            require_once dirname( __FILE__ ) . '/vendor/freemius/start.php';

            $hio_fs = fs_dynamic_init( array(
                'id'                  => '21577',
                'slug'                => 'hibiscus-image-optimizer',
                'type'                => 'plugin',
                'public_key'          => 'pk_706a8e2e29148d4221c3a76773ea7',
                'is_premium'          => false,
                'has_addons'          => false,
                'has_paid_plans'      => true,
                'menu'                => array(
                    'slug'           => 'hibiscus-image-optimizer',
                    'account'        => true,
                    'support'        => true,
                    'parent'         => array(
                        'slug' => 'upload.php',
                    ),
                ),
            ) );
        }

        return $hio_fs;
    }

    // Init Freemius.
    hio_fs();
    // Signal that SDK was initiated.
    do_action( 'hio_fs_loaded' );
}
class HIO_Converter {

    public static $option_name = 'hio_settings';

    // --- Plugin Initialization and Hooks ---

    public function __construct() {
        // Enqueue scripts and styles
             add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        // Add Admin Menu Page
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

        $settings = $this->get_settings();

        // On-Upload Conversion Hook (Conditional based on saved settings)
        if ( ! empty( $settings['on_upload_enabled'] ) ) {
            add_filter( 'wp_generate_attachment_metadata', array( $this, 'convert_on_upload' ), 10, 2 );
        }

        // AJAX Handler for Bulk Conversion
        add_action( 'wp_ajax_hio_bulk_convert', array( $this, 'ajax_bulk_convert' ) );
        // AJAX Handler for Settings Save
        add_action( 'wp_ajax_hio_save_settings', array( $this, 'ajax_save_settings' ) );
       // add_filter( 'the_content', array( $this, 'replace_images_in_content' ), 99 );
        add_filter( 'post_thumbnail_html', array( $this, 'replace_images_in_content' ), 99 );
        add_filter( 'wp_calculate_image_srcset', array( $this, 'add_optimized_formats_to_srcset' ), 10, 5 );
        add_action( 'wp_ajax_hio_check_format_support', array( $this, 'ajax_check_format_support' ) );
        add_action( 'wp_ajax_hio_start_background_optimization', [ $this, 'ajax_start_background_optimization' ] );
        add_action( 'hio_background_optimize_event', [ $this, 'hio_run_background_optimization' ] );
    }

    // --- Settings and UI Rendering ---

    public function get_settings() {
        return get_option( self::$option_name, array(
            'on_upload_enabled' => 0,
            'target_format'     => 'webp', // webp or avif
            'conversion_quality'=> 85,
            'serving_method'    => 'picture',
        ) );
    }
    private function check_conversion_support( $format ) {
        if ( $format === 'webp' ) {
            // 1. Check if the GD extension is loaded
            if ( ! extension_loaded( 'gd' ) ) {
                return 'missing_gd';
            }

            // 2. Check if the GD installation supports WebP functions
            if ( ! function_exists( 'imagewebp' ) ) {
                return 'no_webp_support_gd';
            }

            return 'supported';

        } elseif ( $format === 'avif' ) {
            // 1. Check if Imagick extension is loaded
            if ( ! class_exists( 'Imagick' ) ) {
                return 'missing_imagick';
            }

            try {
                $imagick = new Imagick();
                $formats = $imagick->queryFormats();

                // 2. Check if AVIF is listed in the supported formats
                if ( in_array( 'AVIF', $formats ) ) {
                    return 'supported';
                }
            } catch ( Exception $e ) {
                return 'imagick_error'; // Imagick present but failing
            }

            return 'no_avif_support'; // Imagick present but no AVIF delegate
        }

        return 'unsupported_format_check';
    }
    public function add_admin_menu() {
        add_media_page(
            'Hibiscus Optimizer',
            'Hibiscus Optimizer', // Menu Title
            'manage_options',
            HIO_SLUG, // Menu Slug
            array( $this, 'render_admin_page' )
        );
    }
    public function ajax_check_format_support() {
        // Basic security check
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied.' );
        }

        $format = isset( $_POST['format'] ) ? sanitize_text_field( $_POST['format'] ) : null;
        if ( ! $format ) {
            wp_send_json_error( 'Missing format parameter.' );
        }

        $support_status = $this->check_conversion_support( $format );

        if ( $support_status === 'supported' ) {
            wp_send_json_success( array( 'supported' => true ) );
        } else {
            // Send the error type back to the JavaScript
            wp_send_json_error( array(
                'supported' => false,
                'reason' => $support_status
            ) );
        }
    }
    public function render_admin_page() {
        ?>
        <div class="wrap" id="hio-app">
            <h1 class="wp-heading-inline">Hibiscus Image Optimizer</h1>
            <div id="hio-dashboard">
            </div>
        </div>
        <?php
    }

 public function enqueue_assets( $hook_suffix ) {
    // Only load on your plugin's Media subpage
    if ( $hook_suffix !== 'media_page_' . HIO_SLUG ) {
        return;
    }

    // Enqueue admin assets
    wp_enqueue_style( 'hio-style', HIO_URL . 'admin/assets/css/hio-style.css', array(), HIO_VERSION );
    wp_enqueue_script( 'hio-script', HIO_URL . 'admin/assets/js/hio-script.js', array( 'jquery' ), HIO_VERSION, true );

    // Safely check Freemius (handles both initialized and not-yet cases)
    $is_premium = false;
    if ( function_exists( 'hio_fs' ) ) {
        try {
            $is_premium = hio_fs()->can_use_premium_code();
        } catch ( Exception $e ) {
            error_log( 'Freemius check failed: ' . $e->getMessage() );
        }
    }

    // Localize the data BEFORE the script runs
    wp_localize_script( 'hio-script', 'HIO_Data', array(
        'ajax_url'         => admin_url( 'admin-ajax.php' ),
        'nonce'            => wp_create_nonce( 'hio-nonce' ),
        'initial_settings' => $this->get_settings(),
        'images_to_process'=> $this->count_unconverted_images(),
        'is_premium'       => $is_premium,
    ) );

    //error_log( 'HIO Localized, Premium: ' . ( $is_premium ? 'Yes' : 'No' ) );
}

 public function ajax_start_background_optimization() {
        check_ajax_referer( 'hio-nonce', 'security' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => 'Unauthorized access.' ] );
        }

        // Optional premium check
        if ( function_exists( 'hibiscus_fs' ) && ! hibiscus_fs()->can_use_premium_code() ) {
            wp_send_json_error( [ 'message' => 'Upgrade to unlock background optimization.' ] );
        }

        // Schedule the cron event
        if ( ! wp_next_scheduled( 'hio_background_optimize_event' ) ) {
            wp_schedule_event( time() + 10, 'every_five_minutes', 'hio_background_optimize_event' );
        }

        update_option( 'hio_background_optimization_enabled', true );

        wp_send_json_success( [ 'message' => 'Background optimization has been scheduled successfully! You can safely close this page — progress will continue automatically. Check back later to see your updated image count.' ] );
    }

    public function hio_run_background_optimization() {
    $batch_size = 10;
    $offset     = get_option( 'hio_last_offset', 0 );

    $total_images = $this->count_unconverted_images();
    $settings     = $this->get_settings();
    $format       = $settings['target_format'];
    $quality      = $settings['conversion_quality'];

    // Get next batch
    $image_ids = $this->get_image_batch( $offset, $batch_size );
    $processed_count = 0;

    if ( ! empty( $image_ids ) ) {
        foreach ( $image_ids as $attachment_id ) {
            $file = get_attached_file( $attachment_id );
            $this->convert_file( $file, $format, $quality );

            $metadata = wp_get_attachment_metadata( $attachment_id );
            if ( $metadata ) {
                $this->convert_on_upload( $metadata, $attachment_id );
            }

            $processed_count++;
        }
    }

    $next_offset = $offset + $processed_count;
    $is_finished = ( $total_images > 0 && $next_offset >= $total_images ) || empty( $image_ids );

    if ( $is_finished ) {
        update_option( 'hio_last_offset', 0 ); // Reset offset
        update_option( 'hio_background_optimization_enabled', false );
        //error_log( '✅ HIO Background Optimization Finished' );
    } else {
        update_option( 'hio_last_offset', $next_offset );
        //error_log( "🔁 HIO Optimized batch {$offset} to {$next_offset}" );
    }
    }
    // --- On-Upload Conversion ---

    public function convert_on_upload( $metadata, $attachment_id ) {
        $file = get_attached_file( $attachment_id );
        $settings = $this->get_settings();
        $format = $settings['target_format'];
        $quality = $settings['conversion_quality'];

        // Only process JPEGs and PNGs
        $mime_type = get_post_mime_type( $attachment_id );
        if ( strpos( $mime_type, 'image/jpeg' ) !== false || strpos( $mime_type, 'image/png' ) !== false ) {

            // Convert the full size
            $this->convert_file( $file, $format, $quality );

            // Convert all registered image sizes
            if ( isset( $metadata['sizes'] ) ) {
                $base_dir = trailingslashit( dirname( $file ) );

                foreach ( $metadata['sizes'] as $size_data ) {
                    $sized_file_path = $base_dir . $size_data['file'];
                    $this->convert_file( $sized_file_path, $format, $quality );
                }
            }
        }
        return $metadata;
    }

    // --- Bulk Conversion & Utility Functions ---

 private function count_unconverted_images() {
    // 1. Get target format from plugin settings
    $settings = get_option('hio_settings', []);
    $target_format = !empty($settings['target_format']) ? strtolower($settings['target_format']) : 'webp';

    // 2. Query for all JPG/PNG attachments
    $args = array(
        'post_type'      => 'attachment',
        'post_mime_type' => array('image/jpeg', 'image/png'),
        'post_status'    => 'inherit',
        'posts_per_page' => -1,
        'fields'         => 'ids',
    );

    $query = new WP_Query($args);
    $unconverted_count = 0;

    if (!empty($query->posts)) {
        foreach ($query->posts as $attachment_id) {
            $file_path = get_attached_file($attachment_id);

            // Skip invalid paths
            if (empty($file_path) || !file_exists($file_path)) {
                continue;
            }

            // Normalize directory separators (important for Windows)
            $file_path = wp_normalize_path($file_path);

            // Get file info
            $path_info = pathinfo($file_path);

            // Properly construct the converted file path
            $converted_path = trailingslashit($path_info['dirname']) . $path_info['filename'] . '.' . $target_format;

            // Normalize again (to avoid "C:\" issues)
            $converted_path = wp_normalize_path($converted_path);

            // Debug log
            error_log("Checking: " . $converted_path . " | Exists: " . (file_exists($converted_path) ? 'Yes' : 'No'));

            // Count only if converted file does NOT exist
            if (!file_exists($converted_path)) {
                $unconverted_count++;
            }
        }
    }

    return $unconverted_count;
}




    private function get_image_batch( $offset, $limit = 10 ) {
        $args = array(
            'post_type'      => 'attachment',
            'post_mime_type' => array('image/jpeg', 'image/png'),
            'post_status'    => 'inherit',
            'posts_per_page' => $limit,
            'offset'         => $offset,
            'fields'         => 'ids',
        );
        $query = new WP_Query( $args );
        return $query->posts;
    }


    // The core conversion utility (WebP is widely supported by GD)
    // --- The core conversion utility (WebP is widely supported by GD) ---
// --- The core conversion utility (AVIF enabled with Imagick) ---
    private function convert_file( $source_path, $format = 'webp', $quality = 85 ) {
        if ( ! file_exists( $source_path ) ) return false;
   $path_info = pathinfo($source_path);
    $destination_path = trailingslashit($path_info['dirname']) . $path_info['filename'] . '.' . $format;

    // Prevent overwriting: skip if file already exists
 
    $extension = strtolower($path_info['extension']);
        $extension = strtolower( pathinfo( $source_path, PATHINFO_EXTENSION ) );
       
        // Check if destination file already exists and skip re-conversion
        if ( file_exists( $destination_path ) ) return true;

        // Check for Imagick and AVIF support
        if ( $format === 'avif' && class_exists( 'Imagick' ) ) {
            try {
                $imagick = new Imagick();
                $imagick->readImage( $source_path );
                // AVIF is a lossy format, set the quality
                $imagick->setImageFormat( 'avif' );
                $imagick->setCompressionQuality( $quality );
                $imagick->writeImage( $destination_path );
                $imagick->clear();
                $imagick->destroy();
                return true;
            } catch ( Exception $e ) {
                // Log the error if Imagick conversion fails (e.g., due to missing libavif)
                error_log( "Hibiscus AVIF Conversion Error: " . $e->getMessage() );
                return false;
            }
        }

        // --- Fallback/WebP Conversion (using GD) ---

        if ( $format === 'webp' && ! function_exists( 'imagewebp' ) ) return false;

        $image = false;
        if ( $extension === 'jpg' || $extension === 'jpeg' ) {
            $image = imagecreatefromjpeg( $source_path );
        } elseif ( $extension === 'png' ) {
            $image = imagecreatefrompng( $source_path );

            // FIX for "Palette image not supported"
            if ( imageistruecolor( $image ) === false ) {
                $temp_image = imagecreatetruecolor( imagesx( $image ), imagesy( $image ) );
                imagecopy( $temp_image, $image, 0, 0, 0, 0, imagesx( $image ), imagesy( $image ) );
                imagedestroy( $image );
                $image = $temp_image;
            }

            imagealphablending( $image, false );
            imagesavealpha( $image, true );
        } else {
            return false;
        }

        if ( ! $image ) return false;

        $success = false;
        if ( $format === 'webp' ) {
            $success = imagewebp( $image, $destination_path, $quality );
        }

        imagedestroy( $image );
        return $success;
    }
    // --- AJAX Handlers ---

    public function ajax_save_settings() {
        check_ajax_referer( 'hio-nonce', 'security' );

        $new_settings = array();
        $new_settings['on_upload_enabled']  = isset( $_POST['on_upload_enabled'] ) ? intval( $_POST['on_upload_enabled'] ) : 0;
        $new_settings['target_format']      = sanitize_text_field( $_POST['target_format'] );
        $new_settings['conversion_quality'] = intval( $_POST['conversion_quality'] );
        $new_settings['serving_method']      = isset( $_POST['serving_method'] ) ? sanitize_text_field( $_POST['serving_method'] ) : 'picture';

        update_option( self::$option_name, $new_settings );
$htaccess_result = ['success' => false, 'message' => 'HTACCESS rules not applied.'];
if ($new_settings['serving_method'] === 'htaccess') {
    $htaccess_result = $this->generate_htaccess_rules($new_settings['target_format']);
}
     wp_send_json_success( array(
    'message' => 'Settings saved successfully!',
    'htaccess' => $htaccess_result['message'],
    'settings' => $new_settings,
) );

    }

    public function ajax_bulk_convert() {
        check_ajax_referer( 'hio-nonce', 'security' );

        $offset = isset( $_POST['offset'] ) ? intval( $_POST['offset'] ) : 0;
        $total_images = isset( $_POST['total_images'] ) ? intval( $_POST['total_images'] ) : $this->count_unconverted_images();
        $batch_size = 10;
        $processed_count = 0;
        $settings = $this->get_settings();
        $format = $settings['target_format'];
        $quality = $settings['conversion_quality'];

        // Get batch of image IDs
        $image_ids = $this->get_image_batch( $offset, $batch_size );

        if ( ! empty( $image_ids ) ) {
            foreach ( $image_ids as $attachment_id ) {
                $file = get_attached_file( $attachment_id );
                $this->convert_file( $file, $format, $quality ); // Convert full size

                // Re-run metadata hook to process all thumbnails/sizes
                $metadata = wp_get_attachment_metadata( $attachment_id );
                if( $metadata ) {
                    $this->convert_on_upload( $metadata, $attachment_id );
                }
                $processed_count++;
            }
        }

        $next_offset = $offset + $processed_count;
        $is_finished = ( $total_images > 0 && $next_offset >= $total_images ) || empty( $image_ids );
        $progress = $total_images > 0 ? min( 100, floor( ( $next_offset / $total_images ) * 100 ) ) : 100;

        wp_send_json_success( array(
            'progress' => $progress,
            'offset' => $next_offset,
            'total_images' => $total_images,
            'is_finished' => $is_finished,
            'message' => $is_finished ? 'Finished the optimization process.' : "Processing batch {$offset} to {$next_offset}..."
        ) );
    }
    public function replace_images_in_content( $content ) {
        if ( is_admin() || strpos( $content, '<img' ) === false ) {
            return $content;
        }

        $settings = $this->get_settings();
        $target_format = $settings['target_format'];

        if ( $target_format !== 'webp' && $target_format !== 'avif' ) {
            return $content;
        }
if ( $serving_method === 'picture' ) {
        // Regex to capture the full <img> tag, and specifically the 'src', 'srcset', and 'sizes' attributes.
        $content = preg_replace_callback(
            '/<img\s+([^>]*?)src=["\']([^"\']+)["\']([^>]*?)(srcset=["\']([^"\']+)["\']|)([^>]*?)(\s+sizes=["\']([^"\']+)["\']|)([^>]*?)\/?>/i',
            function( $match ) use ( $target_format ) {

                $original_img_tag = $match[0];
                $original_src = $match[2]; // Captured src URL
                $original_srcset = isset($match[5]) ? $match[5] : ''; // Captured srcset content
                $original_sizes = isset($match[8]) ? $match[8] : ''; // Captured sizes content

                // We need a path check to ensure the files exist
                $base_path = str_replace( WP_CONTENT_URL, WP_CONTENT_DIR, $original_src );

                // --- Parse All Sources from original srcset ---

                $avif_srcset_pairs = [];
                $webp_srcset_pairs = [];

                // Combine the main src URL and all srcset URLs for processing
                $sources_to_process = [];
                $sources_to_process[] = ['url' => $original_src, 'descriptor' => ''];

                if ( $original_srcset ) {
                    $srcset_parts = explode( ',', $original_srcset );
                    foreach( $srcset_parts as $part ) {
                        // Extract URL and descriptor (e.g., '520w' or '2x')
                        if( preg_match( '/^(.+)\s+([0-9.]+x|[0-9]+w)$/', trim( $part ), $size_match ) ) {
                            $sources_to_process[] = [
                                'url' => trim( $size_match[1] ),
                                'descriptor' => trim( $size_match[2] )
                            ];
                        }
                    }
                }

                // --- Build New Srcset for Optimized Formats ---
                foreach ( array_unique($sources_to_process, SORT_REGULAR) as $source ) {
                    $url = $source['url'];
                    $descriptor = $source['descriptor'];

                    // Convert URL to server path to check existence
                    $url_parts = explode('/', $url);
                    $file_name = end($url_parts);

                    // This is a rough path mapping. Use $original_src's directory for reliable pathing.
                    $base_dir = dirname(str_replace(WP_CONTENT_URL, WP_CONTENT_DIR, $original_src));
                    $file_path = path_join($base_dir, $file_name);

                    // Check for optimized versions
                    if ( file_exists( $file_path . '.avif' ) ) {
                        $avif_srcset_pairs[] = esc_url( $url . '.avif' ) . " " . $descriptor;
                    }
                    if ( file_exists( $file_path . '.webp' ) ) {
                        $webp_srcset_pairs[] = esc_url( $url . '.webp' ) . " " . $descriptor;
                    }
                }

                // If no optimized sources were found, return the original tag
                if ( empty( $avif_srcset_pairs ) && empty( $webp_srcset_pairs ) ) {
                    return $original_img_tag;
                }

                // --- Construct the <picture> Element ---

                $picture_tag = '<picture>';
                $sizes_attr = $original_sizes ? ' sizes="' . esc_attr($original_sizes) . '"' : '';

                // 1. AVIF Source (if requested and sources available)
                if ( $target_format === 'avif' && ! empty( $avif_srcset_pairs ) ) {
                    $avif_srcset = implode( ', ', $avif_srcset_pairs );
                    $picture_tag .= '<source srcset="' . esc_attr( $avif_srcset ) . '" type="image/avif"' . $sizes_attr . '>';
                }

                // 2. WebP Source (always include if available, as a fallback to AVIF or the primary format)
                if ( ! empty( $webp_srcset_pairs ) ) {
                    $webp_srcset = implode( ', ', $webp_srcset_pairs );
                    $picture_tag .= '<source srcset="' . esc_attr( $webp_srcset ) . '" type="image/webp"' . $sizes_attr . '>';
                }

                // 3. Fallback: Original <img> tag (must be last)
                $picture_tag .= $original_img_tag;
                $picture_tag .= '</picture>';
$original_img_tag = preg_replace(
    '/<img\s+/',
    '<img data-optimized="true" ',
    $original_img_tag,
    1
);
                return $picture_tag;
            },
            $content
        );
    }
else
{
        return $content;
}
    }
    private function check_avif_conversion_support() {
        // 1. Check if Imagick extension is loaded
        if ( ! class_exists( 'Imagick' ) ) {
            return 'missing_imagick';
        }

        try {
            $imagick = new Imagick();
            $formats = $imagick->queryFormats();

            // 2. Check if AVIF is listed in the supported formats
            if ( in_array( 'AVIF', $formats ) ) {
                return 'supported';
            }
        } catch ( Exception $e ) {
            // Imagick might be present but malfunctioning (e.g., missing dependencies)
            return 'imagick_error';
        }

        return 'no_avif_support';
    }
    /**
 * Handle AJAX request to start background optimization.
 */


    /**
     * Filters the image sources array to add optimized AVIF and WebP sources.
     * This function controls the formats that get output in the <picture> element.
     */
    public function add_optimized_formats_to_srcset( $sources, $size_array, $image_src, $image_meta, $attachment_id ) {

        $settings = $this->get_settings();
        $target_format = $settings['target_format']; // 'avif' or 'webp'

        // Only proceed if the attachment has metadata and a target format is selected
        if ( empty( $sources ) || empty( $image_meta ) || ( $target_format !== 'webp' && $target_format !== 'avif' ) ) {
            return $sources;
        }

        $image_base_dir = dirname( get_attached_file( $attachment_id ) );
        $new_sources = array();

        // --- 1. Generate Optimized Sources ---

        foreach ( $sources as $width => $source_data ) {
            $url = $source_data['url'];
            $descriptor = $source_data['descriptor']; // 'w' or 'x'

            // Extract the filename to check for the optimized version on disk
            $file_name = basename( $url );
            $file_path = path_join( $image_base_dir, $file_name );

            // A. Check for AVIF
            $avif_path = $file_path . '.avif';
            if ( file_exists( $avif_path ) ) {
                $new_sources['avif_' . $width] = array(
                    'url'        => $url . '.avif',
                    'descriptor' => $descriptor,
                    'width'      => $width,
                    'mime-type'  => 'image/avif',
                );
            }

            // B. Check for WebP (WebP is always generated if AVIF is not chosen, or as a general fallback)
            $webp_path = $file_path . '.webp';
            if ( file_exists( $webp_path ) ) {
                $new_sources['webp_' . $width] = array(
                    'url'        => $url . '.webp',
                    'descriptor' => $descriptor,
                    'width'      => $width,
                    'mime-type'  => 'image/webp',
                );
            }
        }

        // --- 2. Order Sources Based on Settings ---

        // The wp_calculate_image_srcset filter now also handles the <picture> output
        // starting in WordPress 6.5. We must return the sources in the desired priority:
        // AVIF > WebP > Original

        $ordered_sources = array();

        // High priority: AVIF (if selected as target format)
        if ( $target_format === 'avif' ) {
            foreach ( $new_sources as $key => $data ) {
                if ( $data['mime-type'] === 'image/avif' ) {
                    $ordered_sources[$key] = $data;
                }
            }
        }

        // Medium priority: WebP (Always include WebP sources if they exist, regardless of the target)
        // This ensures WebP compatibility even if the target was AVIF (for browsers that support WebP but not AVIF).
        foreach ( $new_sources as $key => $data ) {
            if ( $data['mime-type'] === 'image/webp' ) {
                $ordered_sources[$key] = $data;
            }
        }

        // Low priority: Original image sources (JPEGs/PNGs)
        foreach ( $sources as $key => $data ) {
            // Only include the original sources if they are not already in the optimized array keys
            if ( ! isset( $ordered_sources[$key] ) ) {
                $ordered_sources[$key] = $data;
            }
        }

        // The merged and ordered array dictates the final <picture> structure.
        return $ordered_sources;
    }
 private function generate_htaccess_rules($target_format = 'webp') {
    $htaccess_path = ABSPATH . '.htaccess';
if (!is_writable($htaccess_path)) {
    return ['success' => false, 'message' => '.htaccess is not writable. Please check file permissions.'];
}
    $start_marker = "# Hibiscus Optimizer Start\n";
    $end_marker   = "# Hibiscus Optimizer End\n";

    // Build rules string
    $rules = $start_marker . "<IfModule mod_rewrite.c>\nRewriteEngine On\n";

    if ($target_format === 'avif') {
        $rules .= <<<EOD
RewriteCond %{HTTP_ACCEPT} image/avif
RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$
RewriteCond %{REQUEST_FILENAME}.avif -f
RewriteRule (.*)\.(jpe?g|png)$ %{REQUEST_URI}.avif [T=image/avif,L]

EOD;
    }

    // WebP fallback
    $rules .= <<<EOD
RewriteCond %{HTTP_ACCEPT} image/webp
RewriteCond %{REQUEST_FILENAME} (.*)\.(jpe?g|png)$
RewriteCond %{REQUEST_FILENAME}.webp -f
RewriteRule (.*)\.(jpe?g|png)$ %{REQUEST_URI}.webp [T=image/webp,L]

EOD;

    $rules .= "</IfModule>\n" . $end_marker;

    // Read existing .htaccess
    $existing_htaccess = file_exists($htaccess_path) ? file_get_contents($htaccess_path) : '';

    // Remove previous Hibiscus Optimizer block if exists
    $clean_htaccess = preg_replace(
        '/# Hibiscus Optimizer Start.*# Hibiscus Optimizer End\n/sU',
        '',
        $existing_htaccess
    );

    // Merge new rules with existing content
    $final_htaccess = $clean_htaccess . "\n" . $rules;

    // Write to .htaccess
    $written = file_put_contents($htaccess_path, $final_htaccess);

    if ($written !== false) {
        return ['success' => true, 'message' => ".htaccess rules generated/updated successfully for {$target_format}!"];
    } else {
        return ['success' => false, 'message' => "Failed to write .htaccess. Please check file permissions."];
    }
}

}

// Start the Hibiscus Image Optimizer
new HIO_Converter();
// Replace <img> src with the target format
// Replace <img> src with the target format
// 1. Replace src for all attachment images
add_filter('wp_get_attachment_image_attributes', function($attr, $attachment, $size) {
    $settings = get_option('hio_settings', []);
    $target_format = $settings['target_format'] ?? 'webp';

    $file = get_attached_file($attachment->ID);
    if (!$file) return $attr;

    $info = pathinfo($file);
    $new_file = $info['dirname'] . '/' . $info['filename'] . '.' . $target_format;

    if (file_exists($new_file)) {
        $url = str_replace(ABSPATH, site_url('/') , $new_file);
        $attr['src'] = $url;
    }

    return $attr;
}, 10, 3);

// 2. Replace srcset URLs for responsive images
add_filter('wp_calculate_image_srcset', function($sources, $size_array, $image_src, $image_meta, $attachment_id) {
    $settings = get_option('hio_settings', []);
    $target_format = $settings['target_format'] ?? 'webp';

    $file = get_attached_file($attachment_id);
    if (!$file) return $sources;

    $info = pathinfo($file);
    $new_file = $info['dirname'] . '/' . $info['filename'] . '.' . $target_format;

    if (file_exists($new_file)) {
        $url = str_replace(ABSPATH, site_url('/') , $new_file);
        $new_sources = [];
        foreach ($sources as $width => $source) {
            $new_sources[$width] = [
                'url' => $url,
                'descriptor' => $source['descriptor'],
            ];
        }
        return $new_sources;
    }

    return $sources;
}, 10, 5);

// 3. Replace <img> tags in post content for all images
add_filter('the_content', function($content) {
    $settings = get_option('hio_settings', []);
    $target_format = $settings['target_format'] ?? 'webp';

    return preg_replace_callback('/<img[^>]+src=["\']([^"\']+)["\']/i', function($matches) use ($target_format) {
        $src = $matches[1];

        // Get local file path
        $file_path = str_replace(site_url('/'), ABSPATH, $src);

        if (file_exists($file_path)) {
            $info = pathinfo($file_path);
            $new_file = $info['dirname'] . '/' . $info['filename'] . '.' . $target_format;

            if (file_exists($new_file)) {
                $new_url = str_replace(ABSPATH, site_url('/'), $new_file);
                return str_replace($src, $new_url, $matches[0]);
            }
        }

        return $matches[0];
    }, $content);
});
// $hio_instance = new HIO_Converter();

// // Attach AJAX (if not already attached inside the class)
// add_action( 'wp_ajax_hio_start_background_optimization', [ $hio_instance, 'ajax_start_background_optimization' ] );

// // Attach cron using the actual method name that exists on the class
// add_action( 'hio_background_optimize_event', [ $hio_instance, 'hio_run_background_optimization' ] );

add_filter( 'cron_schedules', function( $schedules ) {
    $schedules['every_five_minutes'] = [
        'interval' => 300, // 5 minutes
        'display'  => __( 'Every 5 Minutes' ),
    ];
    return $schedules;
});